import sys

# Увеличение лимита рекурсии для Segment Tree
# sys.setrecursionlimit(200000)

def gcd(a, b):
    while b:
        a, b = b, a % b
    return a

# --- Segment Tree для поиска самой длинной непрерывной последовательности одинаковых значений ---

class Node:
    def __init__(self, max_len=0, pref_len=0, suff_len=0, val_left=None, val_right=None):
        self.max_len = max_len
        self.pref_len = pref_len
        self.suff_len = suff_len
        self.val_left = val_left
        self.val_right = val_right

def merge(L, R, mid_idx, N):
    if L is None: return R
    if R is None: return L
    
    P = Node()
    P.val_left = L.val_left
    P.val_right = R.val_right
    P.max_len = max(L.max_len, R.max_len)
    
    # Слияние в середине
    if L.val_right == R.val_left:
        P.max_len = max(P.max_len, L.suff_len + R.pref_len)
    
    # Вычисление pref_len
    P.pref_len = L.pref_len
    if L.pref_len == mid_idx and L.val_right == R.val_left:
        P.pref_len += R.pref_len
    
    # Вычисление suff_len
    P.suff_len = R.suff_len
    if R.suff_len == (N - mid_idx) and L.val_right == R.val_left:
        P.suff_len += L.suff_len
        
    return P

class SegmentTree:
    def __init__(self, data, N):
        self.N = N
        self.data = data
        self.tree = [None] * (4 * N)
        self.build(1, 0, N - 1)

    def build(self, node_idx, L, R):
        if L == R:
            self.tree[node_idx] = Node(1, 1, 1, self.data[L], self.data[L])
            return
        
        mid = (L + R) // 2
        self.build(2 * node_idx, L, mid)
        self.build(2 * node_idx + 1, mid + 1, R)
        
        self.tree[node_idx] = merge(self.tree[2 * node_idx], self.tree[2 * node_idx + 1], mid - L + 1, R - L + 1)

    def update(self, node_idx, L, R, idx, new_val):
        if L == R:
            self.data[idx] = new_val
            self.tree[node_idx] = Node(1, 1, 1, new_val, new_val)
            return
            
        mid = (L + R) // 2
        if L <= idx <= mid:
            self.update(2 * node_idx, L, mid, idx, new_val)
        else:
            self.update(2 * node_idx + 1, mid + 1, R, idx, new_val)
        
        # Обновление узла после рекурсивного обновления детей
        self.tree[node_idx] = merge(self.tree[2 * node_idx], self.tree[2 * node_idx + 1], mid - L + 1, R - L + 1)

    def query(self, node_idx, L, R, qL, qR):
        if qL > R or qR < L:
            return None
        if qL <= L and R <= qR:
            return self.tree[node_idx]
            
        mid = (L + R) // 2
        res_L = self.query(2 * node_idx, L, mid, qL, qR)
        res_R = self.query(2 * node_idx + 1, mid + 1, R, qL, qR)
        
        # Специальное слияние для запроса (длины берутся из реальных диапазонов)
        if res_L is None: return res_R
        if res_R is None: return res_L
        
        # Для запроса, mid_idx = actual_mid_point - L + 1
        return merge(res_L, res_R, max(0, mid - qL + 1), qR - qL + 1)


# --- Основная логика поиска максимальной длины ---

def get_max_len(N, A, L_tree, R_tree):
    
    # 1. Находим самую длинную возможную длину палиндрома
    # max_k = (N-1) // 2
    
    low = 0
    high = N // 2
    max_k = 0

    while low <= high:
        k = low + (high - low) // 2
        
        # Проверяем, существует ли палиндром радиуса k
        is_possible = False
        
        # Итерируем по всем возможным центрам c
        for c in range(k + 1, N - k + 1): # Центр от k+1 до N-k (1-индексация)
            
            # Условие 3: высота центра должна быть достаточной
            if A[c] < k + 1:
                continue
                
            # Условие 1: L константно на [c-k, c]
            # Запрос длины k+1 на отрезке [c-k-1, c-1] (0-индексация)
            # MaxLenL(c) >= k
            
            # L_tree.data - 0-индексация (c-1, c-2, ...)
            # L_tree.data[c-1] соответствует A[c]
            
            # Запрос для L (длина k+1)
            # От c-k-1 до c-1 (для элементов A)
            L_query = L_tree.query(1, 0, N - 1, c - k - 1, c - 1)
            
            is_L_const = False
            if L_query:
                # Проверяем, что весь запрошенный отрезок имеет одинаковое значение
                # Это гарантируется, если max_len = suff_len = pref_len = k+1
                if L_query.max_len >= k + 1:
                    is_L_const = True
                    
            if not is_L_const:
                continue
            
            # Условие 2: R константно на [c, c+k]
            # R_tree.data - 0-индексация (c, c+1, ...)
            
            # Запрос для R (длина k+1)
            # От c-1 до c+k-1 (для элементов A)
            R_query = R_tree.query(1, 0, N - 1, c - 1, c + k - 1)
            
            is_R_const = False
            if R_query:
                 # Проверяем, что весь запрошенный отрезок имеет одинаковое значение
                if R_query.max_len >= k + 1:
                    is_R_const = True
                    
            if not is_R_const:
                continue
            
            # Если все условия выполнены
            is_possible = True
            break
            
        if is_possible:
            max_k = k
            low = k + 1
        else:
            high = k - 1
            
    return 2 * max_k + 1


def main():
    # Чтение N
    try:
        N_str = sys.stdin.readline()
        if not N_str: return
        N = int(N_str)
        
        # Чтение A (высоты)
        A = [0] + list(map(int, sys.stdin.readline().split())) # 1-индексация A
        
        # Чтение Q (запросов)
        Q = int(sys.stdin.readline())
        
    except Exception:
        return
        
    # L_i = A_i - i
    L = [(A[i] - i) for i in range(1, N + 1)] # 0-индексация L
    # R_i = A_i + i
    R = [(A[i] + i) for i in range(1, N + 1)] # 0-индексация R

    # Построение деревьев
    L_tree = SegmentTree(L, N)
    R_tree = SegmentTree(R, N)
    
    # 1. Начальное состояние
    print(get_max_len(N, A, L_tree, R_tree))

    # 2. Обработка Q запросов
    for _ in range(Q):
        try:
            line = sys.stdin.readline().split()
            if not line: break
            i, x = map(int, line)
        except:
            break

        # Обновление A (1-индексация)
        A[i] = x
        
        # Обновление L (0-индексация)
        L_val = x - i
        L_tree.update(1, 0, N - 1, i - 1, L_val)
        
        # Обновление R (0-индексация)
        R_val = x + i
        R_tree.update(1, 0, N - 1, i - 1, R_val)
        
        # Вывод результата
        print(get_max_len(N, A, L_tree, R_tree))

if __name__ == "__main__":
    main()
