import sys
import math

# Увеличиваем лимит рекурсии для Sparse Table
sys.setrecursionlimit(300000)

MOD = 998244353
MAX_A = 10**7
# Порог для НОК. Если НОК превышает это значение, мы обрезаем его. 
# Это предотвращает работу с чрезмерно большими числами, сохраняя при этом логику сравнения.
M_LCM = 2 * 10**18 

# --- Математические функции ---

def gcd(a, b):
    # Встроенный math.gcd
    return math.gcd(a, b)

def _lcm(a, b):
    """Вычисляет НОК с обрезанием по M_LCM."""
    if a == 0 or b == 0:
        return 0
    
    # Остановка, если одно из чисел уже превысило порог
    if a > M_LCM or b > M_LCM:
        return M_LCM + 1
        
    try:
        # Проверка на потенциальное переполнение до умножения
        # Если a * b // gcd(a, b) > M_LCM
        if M_LCM // a < (b // gcd(a, b)):
            return M_LCM + 1
            
        res = abs(a * b) // gcd(a, b)
        return min(res, M_LCM + 1)
    except OverflowError:
        # На всякий случай, если Python все же выдаст ошибку
        return M_LCM + 1

# --- Sparse Table для Range LCM (R_LCM) ---

class SparseTableLCM:
    """Разреженная таблица для O(1) запросов Range LCM."""
    def __init__(self, A, N):
        self.N = N
        self.A = A # 1-индексация
        if N == 0: return
        self.LOG_MAX = int(math.log2(N)) + 1
        # R_LCM[k][i] = lcm(A[i], A[i+1], ..., A[i + 2^k - 1])
        self.R_LCM = [[0] * (N + 1) for _ in range(self.LOG_MAX)]
        
        # log_table[i] = floor(log2(i))
        self.log_table = [0] * (N + 1)
        for i in range(2, N + 1):
            self.log_table[i] = self.log_table[i // 2] + 1
            
        # k=0: базовые значения R_LCM[0][i] = A[i]
        for i in range(1, N + 1):
            self.R_LCM[0][i] = A[i]
            
        # Заполнение таблицы O(N log N * LCM_cost)
        for k in range(1, self.LOG_MAX):
            power_of_2 = 1 << (k - 1)
            for i in range(1, N + 1 - (1 << k) + 1):
                l_val = self.R_LCM[k-1][i]
                r_val = self.R_LCM[k-1][i + power_of_2]
                self.R_LCM[k][i] = _lcm(l_val, r_val)

    def query(self, L, R):
        """Запрос lcm(A[L]...A[R]) за O(1)."""
        if L > R: return 1
        length = R - L + 1
        k = self.log_table[length]
        power_of_2 = 1 << k
        
        l_val = self.R_LCM[k][L]
        r_val = self.R_LCM[k][R - power_of_2 + 1]
        
        return _lcm(l_val, r_val)

# --- Решатель ---

def solve():
    try:
        input = sys.stdin.read
        data = input().split()
        
        if not data: return
        
        N = int(data[0])
        # A - 1-индексация. A[0] не используется.
        A = [0] + [int(x) for x in data[1:N+1]] 
    except Exception:
        return

    # 1. Предобработка (Префиксный и Суффиксный GCD)
    P = [0] * (N + 2) # P[i] = gcd(A[1]...A[i])
    S = [0] * (N + 2) # S[i] = gcd(A[i]...A[N])
    
    # Расчет P: P[0]=0, gcd(0, x)=x
    for i in range(1, N + 1):
        # gcd(0, x) = x, если x > 0. gcd(0, 0) = 0.
        P[i] = gcd(P[i-1], A[i]) if P[i-1] != 0 else A[i]
        
    # Расчет S: S[N+1]=0, gcd(0, x)=x
    for i in range(N, 0, -1):
        S[i] = gcd(S[i+1], A[i]) if S[i+1] != 0 else A[i]

    # 2. Построение Sparse Table для Range LCM
    if N > 0:
        lcm_st = SparseTableLCM(A, N)
    else:
        print(0)
        return

    total_sum = 0
    
    # 3. Главный цикл по i (начало подмассива LCM: A[i..])
    for i in range(1, N + 1):
        
        G_prefix = P[i-1]
        
        # Начинаем с k = i
        k = i 
        
        # 4. Внешний цикл: Группировка по LCM (L_cur)
        # LCM(A[i..k]) меняется не более O(log A_max) раз
        while k <= N:
            j_start = k
            
            # L_cur = LCM(A[i..j_start]) - вычисляем за O(1)
            L_cur = lcm_st.query(i, j_start)
            
            # --- Бинарный Поиск для j_end: ищем max j_end, где LCM(A[i..j_end]) = L_cur ---
            low = j_start
            high = N
            j_end = j_start
            
            # Бинарный поиск O(log N * LCM_cost)
            while low <= high:
                mid = low + (high - low) // 2
                current_lcm = lcm_st.query(i, mid)
                
                if current_lcm == L_cur:
                    j_end = mid
                    low = mid + 1 # Пытаемся расширить дальше
                else:
                    high = mid - 1 # Слишком далеко, уменьшаем
                    
            # -------------------------------------------------------------------
            
            # G_prime = gcd(G_prefix, L_cur)
            G_prime = gcd(G_prefix, L_cur)
            
            # 5. Внутренний цикл: Группировка по Суффиксному GCD (S_{k+1})
            k_inner = j_start 
            # Suffix GCD меняется не более O(log A_max) раз
            while k_inner <= j_end:
                k_start = k_inner
                
                # S_cur = gcd(A[k+1]...A[N]). S[k+1] - Суффиксный GCD.
                S_cur = S[k_start + 1] 
                
                # Находим конец подблока k_end, где S_{m+1} остается S_cur
                # Используем предварительно вычисленный массив S
                k_end = j_end
                temp_k = k_start + 1
                
                # Ищем следующий индекс m, где S[m+1] != S_cur
                # Ограничиваемся текущим LCM-блоком j_end
                # Эта часть все еще использует линейное сканирование, но в сумме по i
                # общее количество итераций здесь составляет O(N log A_max).
                while temp_k <= j_end and S[temp_k + 1] == S_cur:
                    temp_k += 1
                k_end = temp_k - 1
                
                # Количество элементов k в этом подблоке [k_start, k_end]
                count = k_end - k_start + 1
                
                # Боевая сила: f_val = gcd(G_prime, S_cur)
                f_val = gcd(G_prime, S_cur)
                
                # Добавление к сумме: count * f_val
                total_sum = (total_sum + count * f_val) % MOD
                
                k_inner = k_end + 1
            
            # Следующий LCM-блок начинается сразу после j_end
            k = j_end + 1

    # Вывод
    print(total_sum)

# Установка рекурсивных функций для встроенных math (для Python < 3.5)
if not hasattr(math, 'gcd'):
    def _gcd(a, b):
        while b:
            a, b = b, a % b
            return a
    math.gcd = _gcd

if __name__ == "__main__":
    solve()
