import sys

def count_digits(n):
    """Count digits in decimal representation of n"""
    return len(str(n))

def collatz(x):
    """Standard Collatz function"""
    if x % 2 == 0:
        return x // 2
    else:
        return 3 * x + 1

def estimate_immediate_benefit(x):
    """
    Check if next collatz step significantly reduces digits
    """
    next_val = collatz(x)
    current_digits = count_digits(x)
    next_digits = count_digits(next_val)
    return next_digits < current_digits

def should_use_random(x, step_count, total_cost, max_cost_budget=550):
    """
    Strategic random usage to avoid expensive trajectories

    Use random when:
    1. We're approaching our budget limit
    2. Number has many digits and collatz won't immediately help
    3. We've been stuck for a while
    """

    current_digits = count_digits(x)

    # Don't use random for very small numbers
    if x < 10000:
        return False

    # Don't use random if it would create huge numbers
    if current_digits >= 8:
        return False

    # If we're close to budget limit, be more aggressive with random
    if total_cost > max_cost_budget * 0.7:  # 70% of budget used
        # For expensive cases, try random more frequently
        if current_digits >= 6:
            return step_count % 15 == 0
        elif current_digits >= 5:
            return step_count % 25 == 0

    # If collatz will immediately help (reduce digits), use it
    if estimate_immediate_benefit(x):
        return False

    # For large numbers, use random occasionally
    if current_digits >= 6:
        return step_count % 40 == 0
    elif current_digits >= 5:
        return step_count % 60 == 0

    # For very long sequences, try random
    if step_count > 100:
        return step_count % 30 == 0

    return False

def solve_one_number(x0, max_cost_budget=550):
    """Solve for one starting number x0"""
    x = x0
    step_count = 0
    total_cost = 0

    while x != 1:
        step_count += 1

        # Emergency: if cost is too high, use random more aggressively
        emergency_mode = total_cost > max_cost_budget * 0.85

        # Decide which operation to use
        use_random = should_use_random(x, step_count, total_cost, max_cost_budget)

        # In emergency mode, use random more often for large numbers
        if emergency_mode and count_digits(x) >= 5 and not estimate_immediate_benefit(x):
            use_random = step_count % 8 == 0

        if use_random:
            print("random")
        else:
            print("collatz")

        sys.stdout.flush()

        # Read the result
        result = int(input().strip())

        if result == 0:
            # Out of tokens
            return False
        elif result == 1:
            # Success!
            total_cost += 1
            return True
        else:
            # Continue with new number
            x = result
            total_cost += count_digits(x)

    return True

def main():
    # Read number of test cases
    t = int(input().strip())

    for _ in range(t):
        # Read starting number
        x0 = int(input().strip())

        # Solve this number
        success = solve_one_number(x0)

        if not success:
            # Out of tokens, exit
            break

if __name__ == "__main__":
    main()