#include <iostream>
#include <vector>
#include <math.h>
#include <cmath>
#include <iomanip>
#include <algorithm>
#include <set>
#include <unordered_set>
#include <map>
#include <unordered_map>
#include <iomanip>

#define int long long

using namespace std;
using CT = int;
const double eps = 1e-9;
const double pi = acos(-1);

template<typename T> T sqr(const T& val) { return val * val; }

template<typename T, typename U>
bool equal(T v1, U v2) { return abs(v1 - v2) < eps; }

struct Point
{
	CT x = CT(), y = CT();
	Point(const CT& a = 0, const CT& b = 0) : x(a), y(b) {}
	Point operator+(const Point& p) const { return Point(x + p.x, y + p.y); };
	Point& operator+=(const Point& p) { x += p.x; y += p.y; return *this; }
	Point& operator-=(const Point& p) { x -= p.x; y -= p.y; return *this; }
	Point operator-(const Point& p) const { return Point(x - p.x, y - p.y); };
	CT operator*(const Point& p) const { return x * p.x + y * p.y; };
	Point operator*(CT val) const { return Point(x * val, y * val); };
	CT operator^(const Point& p) const { return x * p.y - y * p.x; };
	bool operator==(const Point& p) const { return equal(x, p.x) && equal(y, p.y); };
	bool operator!=(const Point& p) const { return !(*this == p); };
	Point rotate(double angle) const { return Point(x * cos(angle) - y * sin(angle), x * sin(angle) + y * cos(angle)); };
	friend istream& operator>>(istream& in, Point& p) { in >> p.x >> p.y; return in; }
	friend ostream& operator<<(ostream& out, const Point& p) { out << p.x << ' ' << p.y; return out; }
	bool operator<(const Point& p) const
	{
		return (((*this) ^ p) > 0) || (((*this) ^ p) == 0 && (x * x + y * y < p.x* p.x + p.y * p.y));
	}
};

template<typename T>
double rootBin(T val)
{
	double l = 0, r = 1e18;
	for (int i = 0; i < 150; i++)
	{
		double m = (l + r) / 2;
		if (m * m > val)
			r = m;
		else
			l = m;
	}
	return (l + r) / 2;
}

double distBin(Point a, Point b) { return rootBin(sqr(a.x - b.x) + sqr(a.y - b.y)); }
double dist(Point a, Point b) { return sqrtl(sqr(a.x - b.x) + sqr(a.y - b.y)); }
double sdist(Point a, Point b) { return sqr(a.x - b.x) + sqr(a.y - b.y); }
CT triangSquareX2(Point a, Point b, Point c) { return abs((b - a) ^ (c - a)); }
CT triangSquareNoAbs(Point a, Point b, Point c) { return (b - a) ^ (c - a); }

struct Line
{
	CT a, b, c;
	Line(const CT& _a = CT(), const CT& _b = CT(), const CT& _c = CT()) : a(_a), b(_b), c(_c) {}
	Line(const Point& x, const Point& y) : a(x.y - y.y), b(y.x - x.x), c(-(x.y * -(x.x - y.x) + x.x * (x.y - y.y))) {}
	friend istream& operator>>(istream& in, Line& l) { in >> l.a >> l.b >> l.c; return in; };
};

Point lineInters(const Line& a, const Line& b)
{
	CT dmain = a.a * b.b - a.b * b.a;
	CT dx = a.b * b.c - a.c * b.b;
	CT dy = a.c * b.a - a.a * b.c;
	return { dx / dmain, dy / dmain };
}

Line perp(const Line& l, const Point& p)
{
	return Line(-l.b, l.a, l.b * p.x - l.a * p.y);
}

double dist(const Line& l, const Point& p)
{
	return dist(p, lineInters(l, perp(l, p)));
}

struct Segment
{
	Point a, b;
	Segment(const Point& x = Point(), const Point& y = Point()) : a(x), b(y) {}
};

bool btw(CT val, CT b1, CT b2)
{
	if (b1 > b2)
		swap(b1, b2);
	return val >= b1 && val <= b2;
}

bool pointOnSeg(const Point& p, const Segment& s)
{
	return equal(triangSquareX2(p, s.a, s.b), 0) && btw(p.x, s.a.x, s.b.x) && btw(p.y, s.a.y, s.b.y);
}

double dist(const Point& p, const Segment& s)
{
	if ((p - s.a) * (s.b - s.a) <= 0)
		return dist(p, s.a);
	if ((p - s.b) * (s.a - s.b) <= 0)
		return dist(p, s.b);
	return dist(Line(s.a, s.b), p);
}

bool intersect1d(CT a, CT b, CT c, CT d)
{
	if (a > b)
		swap(a, b);
	if (c > d)
		swap(c, d);
	return max(a, c) < min(b, d) + eps * 0.5;
}

bool intersec(const Segment& a, const Segment& b)
{
	return intersect1d(a.a.x, a.b.x, b.a.x, b.b.x) && intersect1d(a.a.y, a.b.y, b.a.y, b.b.y) &&
		triangSquareNoAbs(a.a, a.b, b.a) * triangSquareNoAbs(a.a, a.b, b.b) <= 0 &&
		triangSquareNoAbs(b.a, b.b, a.a) * triangSquareNoAbs(b.a, b.b, a.b) <= 0;
}

double dist(const Segment& a, const Segment& b)
{
	if (intersec(a, b))
		return 0;
	return min(min(dist(a.a, b), dist(a.b, b)), min(dist(b.a, a), dist(b.b, a)));
}

struct Circle
{
	CT r;
	Point c;
	double square()
	{
		return pi * r * r;
	}
	friend istream& operator>>(istream& in, Circle& c)
	{
		in >> c.c >> c.r;
		return in;
	}
};

double angle(Point A, Point O, Point B)
{
	return acos((A - O) * (B - O) / dist(A, O) / dist(B, O));
}

struct Triangle
{
	Point A, B, C;
	double a, b, c;
	double angA, angB, angC;
	double ha, hb, hc;
	double r;
	double R;
	void calc()
	{
		angA = acos((sqr(c) + sqr(b) - sqr(a)) / (2 * c * b));
		angB = acos((sqr(a) + sqr(c) - sqr(b)) / (2 * a * c));
		angC = acos((sqr(a) + sqr(b) - sqr(c)) / (2 * a * b));
		R = a * 0.5 / sin(angA);
		hc = 0.5 * a * b / R;
		hb = 0.5 * a * c / R;
		ha = 0.5 * b * c / R;
		r = square() / (0.5 * (a + b + c));
	}
	Triangle(const Point& _a, const Point& _b, const Point& _c) : A(_a), B(_b), C(_c), a(dist(B, C)), b(dist(A, C)), c(dist(A, B))
	{
		calc();
	}
	Triangle(double _a, double _b, double _c) : a(_a), b(_b), c(_c)
	{
		calc();
		A = Point(0, 0);
		B = Point(c, 0);
		C = Point(b * cos(angA), b * sin(angA));
	}
	double square()
	{
		return a * ha * 0.5;
	}
	void move(const Point& diff)
	{
		A += diff;
		B += diff;
		C += diff;
	}
	void rotate(double angle) // относительно точки A
	{
		Point bb = B - A, cc = C - A;
		B = bb.rotate(angle) + A;
		C = cc.rotate(angle) + A;
	}
	void mirror()
	{
		A.x = -A.x;
		B.x = -B.x;
		C.x = -C.x;
	}
};

vector<Point> intersect(Circle a, Circle b)
{
	if (a.c.x > b.c.x)
		swap(a, b);
	if (sqr(a.r + b.r) < sdist(a.c, b.c))
		return {};
	if (a.c == b.c)
	{
		if (equal(a.r, b.r))
			return vector<Point>(3, { 1, 1 }); // inf
		return {};
	}
	if (sdist(a.c, b.c) < a.r * a.r)
	{
		if (dist(a.c, b.c) + b.r < a.r)
			return {};
		if (equal(dist(a.c, b.c) + b.r, a.r))
			return { a.c + (b.c - a.c) * a.r * (1.0 / dist(a.c, b.c)) };
	}
	if (sdist(b.c, a.c) < b.r * b.r)
	{
		if (dist(a.c, b.c) + a.r < b.r)
			return {};
		if (equal(dist(a.c, b.c) + a.r, b.r))
			return { b.c + (a.c - b.c) * b.r * (1.0 / dist(a.c, b.c)) };
	}
	if (a.c == b.c && a.r == b.r)
		return { a.c, a.c, a.c };
	if (equal(sqr(a.r + b.r), sdist(a.c, b.c)))
		return {(a.c + (b.c - a.c) * (1.0 * a.r / (a.r + b.r)))};
	Triangle t(b.r, a.r, dist(a.c, b.c));
	Triangle t2 = t;
	t2.C = Point(t.C.x, -t.C.y);
	double angle = atan2(b.c.y - a.c.y, b.c.x - a.c.x);
	t.rotate(angle);
	t2.rotate(angle);
	t.move(a.c);
	t2.move(a.c);
	return { t.C, t2.C };
}

bool onCircle(Point p, Circle c)
{
	return equal(sqr(c.r), sdist(p, c.c));
}

bool inCircle(Point p, Circle c)
{
	return sqr(c.r) > sdist(p, c.c);
}

vector<Point> touchPoints(Circle c, Point p)
{
	if (onCircle(p, c))
		return { p };
	if (inCircle(p, c))
		return {};
	double d = dist(c.c, p);
	Triangle t1(c.r, sqrt((d + c.r) * (d - c.r)), d);
	if (p.x > c.c.x)
		t1.mirror();
	Triangle t2 = t1;
	t2.C.y = -t1.C.y;
	double angle = 0;
	if (c.c.x < p.x)
		angle = atan2(p.y - c.c.y, p.x - c.c.x);
	else
		angle = atan2(c.c.y - p.y, c.c.x - p.x);
	t1.rotate(angle);
	t2.rotate(angle);
	t1.move(p);
	t2.move(p);
	return { t1.C, t2.C };
}

double squareOfIntersect(Circle c1, Circle c2)
{
	if (c1.r < c2.r)
		swap(c1, c2);
	if (sdist(c1.c, c2.c) >= sqr(c2.r + c1.r))
		return 0;
	if (inCircle(c2.c, c1) && dist(c1.c, c2.c) + c2.r <= c1.r)
		return c2.square();
	vector<Point> p = intersect(c1, c2);
	if (p.size() < 2)
		return 0;
	if (p.size() == 3)
		return c1.square();
	double a1 = angle(p[0], c1.c, c2.c), a2 = angle(p[0], c2.c, c1.c);
	return (a1 - sin(2 * a1) * 0.5) * sqr(c1.r) + (a2 - sin(a2 * 2) * 0.5) * sqr(c2.r);
}

vector<Point> hull(vector<Point> p)
{
	Point op = p[0];
	for (auto& i : p)
		if (i.x < op.x || (i.x == op.x && i.y < op.y))
			op = i;
	for (auto& i : p)
		i -= op;
	sort(begin(p), end(p));

	vector<Point> res = { {0, 0} };
	for (auto& i : p)
	{
		if (i == Point(0, 0))
			continue;
		while (res.size() >= 2 && triangSquareNoAbs(res[res.size() - 2], res.back(), i) <= 0)
			res.pop_back();
		res.push_back(i);
	}
	for (auto& i : res)
		i += op;
	return res;
}

signed main() {
	int size;
	cin >> size;
	vector<Point> v(size);
	for (auto& i : v)
		cin >> i;
	pair<int, int> fur = {-1, -1};
	for (int i = 0; i < size; i++) {
		for (int j = i + 1; j < size; j++) {
			if (fur.first == -1 || sdist(v[i], v[j]) > sdist(v[fur.first], v[fur.second]))
				fur = {i, j};
		}
	}
	Point a = v[fur.first], b = v[fur.second];
	Line ab(a, b);
	// cout << ab.a << ' ' << ab.b << ' ' << ab.c << '\n';
	Line l1 = perp(ab, a), l2 = perp(ab, b);
	cout << l1.a << ' ' << l1.b << ' ' << l1.c << '\n' << l2.a << ' ' << l2.b << ' ' << l2.c << '\n';
}