#include <iostream>
#include <vector>
#include <algorithm>
#include <cmath>
using namespace std;

struct Point {
    int x, y;
    bool operator<(const Point& p) const {
        return x < p.x || (x == p.x && y < p.y);
    }
    bool operator==(const Point& p) const {
        return x == p.x && y == p.y;
    }
};

long long cross(const Point& O, const Point& A, const Point& B) {
    return (long long)(A.x - O.x) * (B.y - O.y) - (long long)(A.y - O.y) * (B.x - O.x);
}

vector<Point> convex_hull(vector<Point> points) {
    int n = points.size();
    if (n <= 2) return points;
    
    sort(points.begin(), points.end());
    points.erase(unique(points.begin(), points.end()), points.end());
    n = points.size();
    
    vector<Point> lower, upper;
    
    for (int i = 0; i < n; i++) {
        while (lower.size() >= 2 && cross(lower[lower.size()-2], lower[lower.size()-1], points[i]) <= 0)
            lower.pop_back();
        lower.push_back(points[i]);
    }
    
    for (int i = n - 1; i >= 0; i--) {
        while (upper.size() >= 2 && cross(upper[upper.size()-2], upper[upper.size()-1], points[i]) <= 0)
            upper.pop_back();
        upper.push_back(points[i]);
    }
    
    lower.pop_back();
    upper.pop_back();
    lower.insert(lower.end(), upper.begin(), upper.end());
    return lower;
}

long long gcd(long long a, long long b) {
    a = abs(a); b = abs(b);
    while (b) {
        long long t = b;
        b = a % b;
        a = t;
    }
    return a ? a : 1;
}

long long gcd3(long long a, long long b, long long c) {
    return gcd(gcd(a, b), c);
}

int main() {
    ios_base::sync_with_stdio(false);
    cin.tie(NULL);
    
    int n;
    cin >> n;
    
    vector<Point> points(n);
    for (int i = 0; i < n; i++) {
        cin >> points[i].x >> points[i].y;
    }
    
    vector<Point> hull = convex_hull(points);
    int h = hull.size();
    
    if (h < 2) {
        cout << "1 0 0\n";
        cout << "1 0 0\n";
        return 0;
    }
    
    double max_width = -1;
    long long best_a1, best_b1, best_c1;
    long long best_a2, best_b2, best_c2;
    
    // Перебираем только рёбра выпуклой оболочки и их нормали
    for (int i = 0; i < h; i++) {
        Point p1 = hull[i];
        Point p2 = hull[(i + 1) % h];
        
        long long dx = p2.x - p1.x;
        long long dy = p2.y - p1.y;
        
        // Два направления: ребро и нормаль к нему
        vector<pair<long long, long long>> directions = {
            {dx, dy},      // направление ребра
            {-dy, dx}      // нормаль (перпендикуляр)
        };
        
        for (auto [a, b] : directions) {
            if (a == 0 && b == 0) continue;
            
            long long g = gcd(abs(a), abs(b));
            long long norm_a = a / g;
            long long norm_b = b / g;
            
            long long min_val = LLONG_MAX;
            long long max_val = LLONG_MIN;
            
            for (const auto& p : points) {
                long long val = norm_a * p.x + norm_b * p.y;
                min_val = min(min_val, val);
                max_val = max(max_val, val);
            }
            
            double width = (double)(max_val - min_val) / sqrt(norm_a * norm_a + norm_b * norm_b);
            
            if (width > max_width) {
                max_width = width;
                
                long long c1 = -min_val;
                long long c2 = -max_val;
                
                long long g1 = gcd3(norm_a, norm_b, c1);
                best_a1 = norm_a / g1;
                best_b1 = norm_b / g1;
                best_c1 = c1 / g1;
                
                long long g2 = gcd3(norm_a, norm_b, c2);
                best_a2 = norm_a / g2;
                best_b2 = norm_b / g2;
                best_c2 = c2 / g2;
            }
        }
    }
    
    cout << best_a1 << " " << best_b1 << " " << best_c1 << "\n";
    cout << best_a2 << " " << best_b2 << " " << best_c2 << "\n";
    
    return 0;
}
