#include <iostream>
#include <memory>
#include <random>
#include <chrono>
#include <format>
#include <vector>

const std::string COLLATZ_QUERY = "COLLAtz";
const std::string RANDOM_QUERY = "rAnDoM";

struct Number {
    std::string number;

    Number operator+(const int n) const {
        std::string ans;
        int carry = n;
        auto process = [&](char c) {
            int d = c - '0';
            d += carry;
            ans.push_back(d % 10 + '0');
            carry = d / 10;
        };
        for (char c : number)
            process(c);
        while (carry)
            process('0');
        return Number{.number = ans};
    }

    Number operator*(const int n) const {
        std::string ans;
        int carry = 0;
        auto process = [&](char c) {
            int d = c - '0';
            d *= n;
            d += carry;
            ans.push_back(d % 10 + '0');
            carry = d / 10;
        };
        for (char c : number)
            process(c);
        while (carry)
            process('0');
        return Number{.number = ans};
    }

    int operator%(const int n) const {
        int r = 0;
        for (auto it = number.rbegin(); it != number.rend(); ++it)
            r = (r * 10 + (*it - '0')) % n;
        return r;
    }

    Number operator/(const int n) const {
        std::string ans;
        int r = 0;
        for (auto it = number.rbegin(); it != number.rend(); ++it) {
            r = r * 10 + (*it - '0');
            if (r / n > 0 || !ans.empty())
                ans.push_back('0' + r / n);
            r %= n;
        }
        std::reverse(ans.begin(), ans.end());
        return Number{.number = ans};
    }

    bool operator<(const Number &other) const {
        if (number.size() != other.number.size())
            return number.size() < other.number.size();
        for (ssize_t i = ssize(number) - 1; i >= 0; --i)
            if (number[i] != other.number[i])
                return number[i] < other.number[i];
        return false;
    }

    template<class Int>
    static Number fromInt(const Int& n) {
        Number ans{
            .number = std::to_string(n),
        };
        std::reverse(ans.number.begin(), ans.number.end());
        return ans;
    }
};

std::ostream& operator<<(std::ostream& out, const Number& n) {
    if (n.number.empty())
        return out << '0';
    for (ssize_t i = ssize(n.number) - 1; i >= 0; --i)
        out << n.number[i];
    return out;
}

struct Strategy {
    virtual void restart() = 0;
    virtual bool choose_collatz(const Number& n) = 0;
    virtual std::string get_name() const = 0;
};

bool interact_test(std::shared_ptr<Strategy> strategy) {
    int64_t n;
    std::cin >> n;
    Number x = Number::fromInt(n);
    static const Number ONE = Number::fromInt(1);
    while (ONE < x) {
        bool choice = strategy->choose_collatz(x);
        std::cout << (choice ? COLLATZ_QUERY : RANDOM_QUERY) << std::endl;
        std::cin >> x.number;
        if (x.number == "0")
            return false;
        std::reverse(x.number.begin(), x.number.end());
    }
    return true;
}

void interact_multitest(std::shared_ptr<Strategy> strategy) {
    size_t t;
    std::cin >> t;
    for (size_t test_index = 1; test_index <= t; ++test_index) {
        strategy->restart();
        if (!interact_test(strategy))
            return;
    }
}

template<class Float>
struct RuntimeOptimalStrategy : Strategy {
    std::vector<Float> f;
    std::vector<bool> c;
    size_t steps;
    Float put_starting_value(size_t n) {
        if (n <= 1)
            return f[n] = 0;
        if (n >= f.size())
            return 1 + put_starting_value(n % 2 ? n * 3 + 1 : n / 2);
        if (f[n])
            return f[n];
        return f[n] = 1 + put_starting_value(n % 2 ? n * 3 + 1 : n / 2);
    }
    void relax(size_t n, Float x, bool collatz) {
        if (f[n] > x) {
            f[n] = x;
            c[n] = collatz;
        }
    }
    RuntimeOptimalStrategy(size_t N, size_t steps) : f(N), c(N, true), steps(steps) {
        for (size_t step_index = 0; step_index < 1 && step_index < steps; ++step_index)
            for (size_t i = 0; i < N; ++i)
                put_starting_value(i);
        for (size_t step_index = 1; step_index < steps; ++step_index) {
            Float window_sum = 0;
            for (size_t i = 1; i < (N + 5) / 6; ++i) {
                if (i % 2 == 0)
                    relax(i, 1 + f[i / 2], true);
                else
                    relax(i, 1 + f[i * 3 + 1], true);
                // window sum is now sum f[3*i-2]..f[6*i-6]
                window_sum -= f[3 * i - 2];
                window_sum -= f[3 * i - 1];
                window_sum -= f[3 * i - 1];
                window_sum += f[6 * i - 5];
                window_sum += f[6 * i - 4];
                window_sum += f[6 * i - 3];
                window_sum += f[6 * i - 2];
                window_sum += f[6 * i - 1];
                window_sum += f[6 * i];
                relax(i, window_sum / (3 * i), false);
            }
            // for the rest for the numbers, only collatz makes sense
            for (size_t i = (N + 5) / 6; i < N; ++i) {
                f[i] = 0;
                put_starting_value(i);
            }
        }
    }
    void restart() override {}
    bool choose_collatz(const Number& n) override {
        return n < Number::fromInt(f.size()) ? c[n % f.size()] : true;
    }
    std::string get_name() const override {
        return std::format("runtime_optimal({}, {})", f.size(), steps);
    }
};

int main(int argc, char* argv[]) {
    std::shared_ptr<Strategy> strategy = std::make_shared<RuntimeOptimalStrategy<float>>(1e8, 1);
    interact_multitest(strategy);
    return 0;
}
