#include <iostream>
#include <memory>
#include <random>
#include <chrono>
#include <format>
#include <mutex>
#include <thread>

const std::string COLLATZ_QUERY = "COLLAtz";
const std::string RANDOM_QUERY = "rAnDoM";

struct Number {
    std::string number;

    Number operator+(const int n) const {
        std::string ans;
        int carry = n;
        auto process = [&](char c) {
            int d = c - '0';
            d += carry;
            ans.push_back(d % 10 + '0');
            carry = d / 10;
        };
        for (char c : number)
            process(c);
        while (carry)
            process('0');
        return Number{.number = ans};
    }

    Number operator*(const int n) const {
        std::string ans;
        int carry = 0;
        auto process = [&](char c) {
            int d = c - '0';
            d *= n;
            d += carry;
            ans.push_back(d % 10 + '0');
            carry = d / 10;
        };
        for (char c : number)
            process(c);
        while (carry)
            process('0');
        return Number{.number = ans};
    }

    int operator%(const int n) const {
        int r = 0;
        for (auto it = number.rbegin(); it != number.rend(); ++it)
            r = (r * 10 + (*it - '0')) % n;
        return r;
    }

    Number operator/(const int n) const {
        std::string ans;
        int r = 0;
        for (auto it = number.rbegin(); it != number.rend(); ++it) {
            r = r * 10 + (*it - '0');
            if (r / n > 0 || !ans.empty())
                ans.push_back('0' + r / n);
            r %= n;
        }
        std::reverse(ans.begin(), ans.end());
        return Number{.number = ans};
    }

    bool operator<(const Number &other) const {
        if (number.size() != other.number.size())
            return number.size() < other.number.size();
        for (ssize_t i = ssize(number) - 1; i >= 0; --i)
            if (number[i] != other.number[i])
                return number[i] < other.number[i];
        return false;
    }

    template<class Int>
    static Number fromInt(const Int& n) {
        Number ans{
            .number = std::to_string(n),
        };
        std::reverse(ans.number.begin(), ans.number.end());
        return ans;
    }
};

std::ostream& operator<<(std::ostream& out, const Number& n) {
    if (n.number.empty())
        return out << '0';
    for (ssize_t i = ssize(n.number) - 1; i >= 0; --i)
        out << n.number[i];
    return out;
}

struct Strategy {
    virtual void restart() = 0;
    virtual bool choose_collatz(const Number& n) = 0;
    virtual std::string get_name() const = 0;
};

bool interact_test(std::shared_ptr<Strategy> strategy) {
    int64_t n;
    std::cin >> n;
    Number x = Number::fromInt(n);
    static const Number ONE = Number::fromInt(1);
    while (ONE < x) {
        bool choice = strategy->choose_collatz(x);
        std::cout << (choice ? COLLATZ_QUERY : RANDOM_QUERY) << std::endl;
        std::cin >> x.number;
        if (x.number == "0")
            return false;
        std::reverse(x.number.begin(), x.number.end());
    }
    return true;
}

void interact_multitest(std::shared_ptr<Strategy> strategy) {
    size_t t;
    std::cin >> t;
    for (size_t test_index = 1; test_index <= t; ++test_index) {
        strategy->restart();
        if (!interact_test(strategy))
            return;
    }
}

struct LogarithmicStrategy : Strategy {
    static size_t simple_kollatz_length(Number n, size_t max_tries) {
        size_t ans;
        const Number ONE = Number::fromInt(1);
        for (ans = 0; ans <= max_tries && ONE < n; ++ans) {
            ++ans;
            if (n % 2 == 0)
                n = n / 2;
            else
                n = n * 3 + 1;
        }
        return ans;
    }
    double d;
    LogarithmicStrategy(const double d) : d(d) {
    }
    void restart() override {}
    bool choose_collatz(const Number& n) override {
        std::string s = n.number + '.' + '0';
        std::reverse(s.begin(), s.end());
        double logarithm = (n.number.size() * log(10) + log(atof(s.c_str()))) * d;
        return simple_kollatz_length(n, floor(logarithm)) <= logarithm || logarithm >= 2000;
    }
    std::string get_name() const override {
        return std::format("logarithmic({:.3f})", d);
    }
};

int main(int argc, char* argv[]) {
    std::shared_ptr<Strategy> strategy = std::make_shared<LogarithmicStrategy>(25.2);
    interact_multitest(strategy);
    return 0;
}
