#include <iostream>
#include <vector>
#include <queue>
#include <cmath>
#include <climits> // For LLONG_MAX
#include <algorithm> // For min, max
#include <unordered_map> // <-- ADDED THIS INCLUDE

using namespace std;

// Use long long for coordinates and costs
typedef long long ll;

// State for our priority queue
// {cost, {location_index, key_mask}}
// location_index 0 = kate's start
// location_index i+1 = key_pos[i]
using State = pair<ll, pair<int, int>>;

/**
* @brief Checks if the path between two points is blocked by any gate
* for which we don't have the key.
*
* @param p1 Starting coordinate.
* @param p2 Ending coordinate.
* @param mask Bitmask of keys Kate has.
* @param n Number of gates/keys.
* @param gate_pos Vector of gate positions.
* @return true if the path is blocked, false otherwise.
*/
bool is_blocked(ll p1, ll p2, int mask, int n, const vector<ll>& gate_pos) {
ll mn = min(p1, p2);
ll mx = max(p1, p2);

for (int i = 0; i < n; ++i) {
// Check if we *don't* have the key for gate i
if (!(mask & (1 << i))) {
// Check if the gate is strictly between the two points
if (gate_pos[i] > mn && gate_pos[i] < mx) {
return true; // Path is blocked
}
}
}
return false; // Path is not blocked
}

/**
* @brief Solves a single test case for Kate's escape.
*/
ll solve() {
int n;
ll kate_pos, hatch_pos;
cin >> n >> kate_pos >> hatch_pos;

vector<ll> key_pos(n);
vector<ll> gate_pos(n);

for (int i = 0; i < n; ++i) {
cin >> key_pos[i] >> gate_pos[i];
}

// --- MEMORY FIX ---
// Original: vector<vector<ll>> dist(n + 1, vector<ll>(1 << n, LLONG_MAX));
// New: An array of n+1 hash maps. dist[loc_idx] maps mask -> cost
vector<unordered_map<int, ll>> dist(n + 1);
// --- END FIX ---


// Min-priority queue: {cost, {loc_idx, mask}}
priority_queue<State, vector<State>, greater<State>> pq;

// Start state: at kate's position (index 0), with 0 keys, at cost 0
dist[0][0] = 0; // This inserts {0, 0} into dist[0] map
pq.push({0, {0, 0}});

while (!pq.empty()) {
auto [d, state] = pq.top();
auto [u_idx, u_mask] = state;
pq.pop();

// --- FIX for stale PQ entries ---
// If we found a shorter path to this state already, skip
// We must check if the key exists (it should) and if d is larger.
if (!dist[u_idx].count(u_mask) || d > dist[u_idx][u_mask]) {
continue;
}
// --- END FIX ---

// Get the actual coordinate of the current location
ll u_pos = (u_idx == 0) ? kate_pos : key_pos[u_idx - 1];

// Try to move to every key 'i' that we don't have yet
for (int i = 0; i < n; ++i) {
// If key 'i' is not in our mask
if (!(u_mask & (1 << i))) {
int v_idx = i + 1; // The location index for key 'i'
ll v_pos = key_pos[i];
int v_mask = u_mask | (1 << i); // New mask after picking up key

// Check if the path to this new key is valid (not blocked)
if (!is_blocked(u_pos, v_pos, u_mask, n, gate_pos)) {
ll new_dist = d + abs(v_pos - u_pos);

// --- FIX for map access ---
// Get the current known distance to this new state
ll current_v_dist = LLONG_MAX;
if (dist[v_idx].count(v_mask)) {
current_v_dist = dist[v_idx][v_mask];
}

// If this is a shorter path, update it
if (new_dist < current_v_dist) {
dist[v_idx][v_mask] = new_dist; // Inserts or updates
pq.push({new_dist, {v_idx, v_mask}});
}
// --- END FIX ---
}
}
}
}

// --- Dijkstra is complete, now find the answer ---

// The final answer is the minimum cost to reach the hatch from *any* state.
ll min_cost = LLONG_MAX;

// 1. Check direct path from start (with 0 keys)
if (!is_blocked(kate_pos, hatch_pos, 0, n, gate_pos)) {
min_cost = abs(hatch_pos - kate_pos);
}

// 2. Check path from Kate's start *after* picking up keys
// --- FIX for map access ---
// Check if a state (0, mask) was ever reached
for (auto const& [mask, d] : dist[0]) { // Iterate over all reached states at loc 0
if (!is_blocked(kate_pos, hatch_pos, mask, n, gate_pos)) {
ll total_cost = d + abs(hatch_pos - kate_pos);
min_cost = min(min_cost, total_cost);
}
}
// --- END FIX ---

// 3. Check path from every key location state
for (int i = 0; i < n; ++i) { // For each key location
int loc_idx = i + 1;
ll pos = key_pos[i];

// --- FIX for map access ---
// Iterate over all masks reached at this location
for (auto const& [mask, d] : dist[loc_idx]) {
// If this state (at key 'i' with 'mask') is reachable
// (d is guaranteed != LLONG_MAX because it's in the map)
// Check if the final path from here to the hatch is clear
if (!is_blocked(pos, hatch_pos, mask, n, gate_pos)) {
ll total_cost = d + abs(hatch_pos - pos);
min_cost = min(min_cost, total_cost);
}
}
// --- END FIX ---
}

if (min_cost == LLONG_MAX) {
return -1; // No path found
} else {
return min_cost;
}
}

int main() {
// Fast I/O
ios_base::sync_with_stdio(false);
cin.tie(NULL);

int t;
cin >> t;
while (t--) {
cout << solve() << "\n";
}
return 0;
}


