import sys
import math


class Point:
    def __init__(self, x, y):
        self.x = x
        self.y = y

    def __sub__(self, other):
        return Point(self.x - other.x, self.y - other.y)

    def __eq__(self, other):
        return self.x == other.x and self.y == other.y

    def cross(self, other):
        return self.x * other.y - self.y * other.x

    def dot(self, other):
        return self.x * other.x + self.y * other.y

    def length2(self):
        return self.x * self.x + self.y * self.y

    def __str__(self):
        return f"({self.x}, {self.y})"


def convex_hull(points):
    """Алгоритм Эндрю для построения выпуклой оболочки"""
    if len(points) <= 1:
        return points

    points = sorted(points, key=lambda p: (p.x, p.y))

    lower = []
    for p in points:
        while len(lower) >= 2 and (lower[-1] - lower[-2]).cross(p - lower[-1]) <= 0:
            lower.pop()
        lower.append(p)

    upper = []
    for p in reversed(points):
        while len(upper) >= 2 and (upper[-1] - upper[-2]).cross(p - upper[-1]) <= 0:
            upper.pop()
        upper.append(p)

    return lower[:-1] + upper[:-1]


def rotating_calipers(hull):
    """Алгоритм вращающихся калиперов для нахождения диаметра"""
    n = len(hull)
    if n == 1:
        return 0, 0
    if n == 2:
        return 0, 1

    k = 1
    while (hull[1] - hull[0]).cross(hull[(k + 1) % n] - hull[0]) > \
            (hull[1] - hull[0]).cross(hull[k] - hull[0]):
        k += 1

    res_i, res_j = 0, k
    for i in range(n):
        while (hull[(i + 1) % n] - hull[i]).cross(hull[(k + 1) % n] - hull[i]) > \
                (hull[(i + 1) % n] - hull[i]).cross(hull[k] - hull[i]):
            k = (k + 1) % n

        if (hull[i] - hull[k]).length2() > (hull[res_i] - hull[res_j]).length2():
            res_i, res_j = i, k

    return res_i, res_j


def line_from_points(p1, p2):
    """Прямая через две точки: ax + by + c = 0"""
    a = p2.y - p1.y
    b = p1.x - p2.x
    c = p2.x * p1.y - p1.x * p2.y

    g = math.gcd(math.gcd(a, b), c)
    if g != 0:
        a //= g
        b //= g
        c //= g

    if a < 0 or (a == 0 and b < 0):
        a, b, c = -a, -b, -c

    return a, b, c


def solve():
    input_data = sys.stdin.read().strip().split()
    n = int(input_data[0])
    points = []

    for i in range(n):
        x = int(input_data[2 * i + 1])
        y = int(input_data[2 * i + 2])
        points.append(Point(x, y))

    if n == 2:
        p1, p2 = points[0], points[1]
        dx, dy = p2.x - p1.x, p2.y - p1.y

        a1, b1, c1 = -dy, dx, dy * p1.x - dx * p1.y

        a2, b2, c2 = -dy, dx, dy * p2.x - dx * p2.y

        g1 = math.gcd(math.gcd(a1, b1), c1)
        if g1 != 0:
            a1, b1, c1 = a1 // g1, b1 // g1, c1 // g1

        g2 = math.gcd(math.gcd(a2, b2), c2)
        if g2 != 0:
            a2, b2, c2 = a2 // g2, b2 // g2, c2 // g2

        if a1 < 0 or (a1 == 0 and b1 < 0):
            a1, b1, c1 = -a1, -b1, -c1
        if a2 < 0 or (a2 == 0 and b2 < 0):
            a2, b2, c2 = -a2, -b2, -c2

        print(a1, b1, c1)
        print(a2, b2, c2)
        return

    hull = convex_hull(points)

    if len(hull) == 1:
        print("1 0 0")
        print("1 0 -1")
        return

    i, j = rotating_calipers(hull)
    p1, p2 = hull[i], hull[j]

    dx, dy = p2.x - p1.x, p2.y - p1.y

    if dx == 0 and dy == 0:
        print("1 0 0")
        print("1 0 -1")
        return

    max_dist = -float('inf')
    min_dist = float('inf')
    max_point = None
    min_point = None

    length = math.sqrt(dx * dx + dy * dy)
    nx, ny = -dy / length, dx / length

    for p in points:
        dist = (p.x - p1.x) * nx + (p.y - p1.y) * ny

        if dist > max_dist:
            max_dist = dist
            max_point = p
        if dist < min_dist:
            min_dist = dist
            min_point = p

    a, b = -dy, dx

    c1 = - (a * min_point.x + b * min_point.y)

    c2 = - (a * max_point.x + b * max_point.y)

    g1 = math.gcd(math.gcd(a, b), c1)
    if g1 != 0:
        a1, b1, c1 = a // g1, b // g1, c1 // g1
    else:
        a1, b1, c1 = a, b, c1

    g2 = math.gcd(math.gcd(a, b), c2)
    if g2 != 0:
        a2, b2, c2 = a // g2, b // g2, c2 // g2
    else:
        a2, b2, c2 = a, b, c2

    if a1 < 0 or (a1 == 0 and b1 < 0):
        a1, b1, c1 = -a1, -b1, -c1
    if a2 < 0 or (a2 == 0 and b2 < 0):
        a2, b2, c2 = -a2, -b2, -c2

    print(a1, b1, c1)
    print(a2, b2, c2)


if __name__ == "__main__":
    solve()