#include <iostream>
#include <vector>
#include <algorithm>
#include <cmath>

using namespace std;

struct Point {
    int x, y;
    Point(int x = 0, int y = 0) : x(x), y(y) {}
    
    bool operator<(const Point& other) const {
        return x < other.x || (x == other.x && y < other.y);
    }
};

long long distSq(const Point& p1, const Point& p2) {
    long long dx = p1.x - p2.x;
    long long dy = p1.y - p2.y;
    return dx * dx + dy * dy;
}

long long cross(const Point& O, const Point& A, const Point& B) {
    return (long long)(A.x - O.x) * (B.y - O.y) - (long long)(A.y - O.y) * (B.x - O.x);
}

vector<Point> convexHull(vector<Point> points) {
    int n = points.size();
    if (n <= 1) return points;
    
    sort(points.begin(), points.end());
    
    vector<Point> hull;
    for (int i = 0; i < n; ++i) {
        while (hull.size() >= 2 && cross(hull[hull.size()-2], hull.back(), points[i]) <= 0)
            hull.pop_back();
        hull.push_back(points[i]);
    }
    
    int lower_size = hull.size();
    for (int i = n - 2; i >= 0; --i) {
        while (hull.size() > lower_size && cross(hull[hull.size()-2], hull.back(), points[i]) <= 0)
            hull.pop_back();
        hull.push_back(points[i]);
    }
    
    hull.pop_back();
    return hull;
}

void findFarthestParallelLines(const vector<Point>& hull) {
    int n = hull.size();
    if (n == 2) {
        Point p1 = hull[0], p2 = hull[1];
        int dx = p2.x - p1.x;
        int dy = p2.y - p1.y;
        
        int a = -dy;
        int b = dx;
        int c1 = -(a * p1.x + b * p1.y);
        int c2 = -(a * p2.x + b * p2.y);
        
        cout << a << " " << b << " " << c1 << "\n";
        cout << a << " " << b << " " << c2 << "\n";
        return;
    }
    
    double maxDist = -1;
    int best_i = 0, best_j = 0;
    
    int left = 0, right = 0;
    for (int i = 1; i < n; i++) {
        if (hull[i].x < hull[left].x) left = i;
        if (hull[i].x > hull[right].x) right = i;
    }
    
    best_i = left;
    best_j = right;
    Point vec(0, -1);
    
    for (int i = 0; i < n; i++) {
        Point p1 = hull[left];
        Point p2 = hull[(left + 1) % n];
        Point p3 = hull[right];
        Point p4 = hull[(right + 1) % n];
        
        Point v1(p2.x - p1.x, p2.y - p1.y);
        Point v2(p4.x - p3.x, p4.y - p3.y);
        
        if (cross(Point(0, 0), v1, v2) >= 0) {
            left = (left + 1) % n;
        } else {
            right = (right + 1) % n;
        }
        
        Point leftPt = hull[left];
        Point rightPt = hull[right];
        
        int dx = hull[(left + 1) % n].x - hull[left].x;
        int dy = hull[(left + 1) % n].y - hull[left].y;
        
        double a = -dy;
        double b = dx;
        double norm = sqrt(a * a + b * b);
        a /= norm;
        b /= norm;
        
        double dist = abs(a * (rightPt.x - leftPt.x) + b * (rightPt.y - leftPt.y));
        
        if (dist > maxDist) {
            maxDist = dist;
            best_i = left;
            best_j = right;
        }
    }
    
    Point p1 = hull[best_i];
    Point p2 = hull[(best_i + 1) % n];
    Point p3 = hull[best_j];
    
    int dx = p2.x - p1.x;
    int dy = p2.y - p1.y;
    
    int a = -dy;
    int b = dx;
    
    int c1 = -(a * p1.x + b * p1.y);
    int c2 = -(a * p3.x + b * p3.y);
    
    bool allBetween = true;
    for (const Point& p : hull) {
        double val = a * p.x + b * p.y;
        if (!(min(c1, c2) <= -val && -val <= max(c1, c2))) {
            allBetween = false;
            break;
        }
    }
    
    if (!allBetween) {
        swap(p1, p3);
        c1 = -(a * p1.x + b * p1.y);
        c2 = -(a * p3.x + b * p3.y);
    }
    
    cout << a << " " << b << " " << c1 << "\n";
    cout << a << " " << b << " " << c2 << "\n";
}

int main() {
    ios_base::sync_with_stdio(false);
    cin.tie(NULL);
    int n;
    cin >> n;
    vector<Point> points(n);
    for (int i = 0; i < n; i++) {
        cin >> points[i].x >> points[i].y;
    }
    vector<Point> hull = convexHull(points);
    findFarthestParallelLines(hull);
    
    return 0;
}