#include <iostream>
#include <vector>
#include <algorithm>
#include <cmath>
using namespace std;

typedef long long ll;

struct Point {
    ll x, y;
    Point() {}
    Point(ll x_, ll y_) : x(x_), y(y_) {}
    Point operator - (const Point& p) const { return Point(x - p.x, y - p.y); }
    Point operator + (const Point& p) const { return Point(x + p.x, y + p.y); }
    ll operator * (const Point& p) const { return x * p.y - y * p.x; } // cross
    ll dot(const Point& p) const { return x * p.x + y * p.y; }
    double len() const { return sqrt(x*x + y*y); }
    bool operator < (const Point& p) const {
        return x < p.x || (x == p.x && y < p.y);
    }
};

ll cross(const Point& a, const Point& b, const Point& c) {
    return (b - a) * (c - a);
}

vector<Point> convexHull(vector<Point> p) {
    int n = p.size();
    if (n <= 1) return p;
    sort(p.begin(), p.end());
    vector<Point> h(n + 1);
    int k = 0;
    for (int i = 0; i < n; i++) {
        while (k >= 2 && cross(h[k-2], h[k-1], p[i]) <= 0) k--;
        h[k++] = p[i];
    }
    int t = k;
    for (int i = n - 2; i >= 0; i--) {
        while (k > t && cross(h[k-2], h[k-1], p[i]) <= 0) k--;
        h[k++] = p[i];
    }
    h.resize(k - 1);
    return h;
}

int main() {
    int n;
    cin >> n;
    vector<Point> p(n);
    for (int i = 0; i < n; i++) {
        cin >> p[i].x >> p[i].y;
    }
    vector<Point> hull = convexHull(p);
    int m = hull.size();
    if (m == 1) {
        // все точки на одной точке - расстояние 0, любые параллельные прямые через эту точку
        cout << "1 0 0\n1 0 0\n";
        return 0;
    }
    if (m == 2) {
        // выпуклая оболочка - отрезок, максимальное расстояние между прямыми = расстояние между точками
        // но прямые должны быть перпендикулярны отрезку
        Point vec = hull[1] - hull[0];
        Point normal(-vec.y, vec.x);
        ll a = normal.x, b = normal.y;
        ll c1 = -(a * hull[0].x + b * hull[0].y);
        ll c2 = -(a * hull[1].x + b * hull[1].y);
        cout << a << " " << b << " " << c1 << "\n";
        cout << a << " " << b << " " << c2 << "\n";
        return 0;
    }

    double maxWidth = -1;
    ll bestA, bestB, bestC1, bestC2;

    int k = 1;
    for (int i = 0; i < m; i++) {
        int ni = (i + 1) % m;
        Point edge = hull[ni] - hull[i];
        Point normal(-edge.y, edge.x);
        ll minProj = normal.dot(hull[0]);
        ll maxProj = minProj;
        for (int j = 1; j < m; j++) {
            ll proj = normal.dot(hull[j]);
            if (proj < minProj) minProj = proj;
            if (proj > maxProj) maxProj = proj;
        }
        double dist = (double)(maxProj - minProj) / normal.len();
        if (dist > maxWidth) {
            maxWidth = dist;
            bestA = normal.x;
            bestB = normal.y;
            bestC1 = -minProj;
            bestC2 = -maxProj;
        }
    }

    cout << bestA << " " << bestB << " " << bestC1 << "\n";
    cout << bestA << " " << bestB << " " << bestC2 << "\n";
    return 0;
}
