#include <iostream>
#include <vector>
#include <algorithm>
#include <cmath>
#include <set>
using namespace std;

struct Point {
    long long x, y;
    bool operator<(const Point& p) const {
        return x < p.x || (x == p.x && y < p.y);
    }
    bool operator==(const Point& p) const {
        return x == p.x && y == p.y;
    }
};

long long cross(const Point& O, const Point& A, const Point& B) {
    return (A.x - O.x) * (B.y - O.y) - (A.y - O.y) * (B.x - O.x);
}

vector<Point> convex_hull(vector<Point> points) {
    int n = points.size();
    if (n <= 2) return points;
    
    sort(points.begin(), points.end());
    points.erase(unique(points.begin(), points.end()), points.end());
    n = points.size();
    
    vector<Point> lower, upper;
    for (int i = 0; i < n; i++) {
        while (lower.size() >= 2 && cross(lower[lower.size()-2], lower[lower.size()-1], points[i]) <= 0)
            lower.pop_back();
        lower.push_back(points[i]);
    }
    
    for (int i = n - 1; i >= 0; i--) {
        while (upper.size() >= 2 && cross(upper[upper.size()-2], upper[upper.size()-1], points[i]) <= 0)
            upper.pop_back();
        upper.push_back(points[i]);
    }
    
    lower.pop_back();
    upper.pop_back();
    lower.insert(lower.end(), upper.begin(), upper.end());
    return lower;
}

long long gcd(long long a, long long b) {
    a = abs(a); b = abs(b);
    while (b) {
        long long t = b;
        b = a % b;
        a = t;
    }
    return a;
}

long long gcd3(long long a, long long b, long long c) {
    return gcd(gcd(a, b), c);
}

struct Normal {
    long long a, b;
    bool operator<(const Normal& n) const {
        return a < n.a || (a == n.a && b < n.b);
    }
};

int main() {
    ios_base::sync_with_stdio(false);
    cin.tie(NULL);
    
    int n;
    cin >> n;
    
    vector<Point> points(n);
    for (int i = 0; i < n; i++) {
        cin >> points[i].x >> points[i].y;
    }
    
    vector<Point> hull = convex_hull(points);
    int h = hull.size();
    
    set<Normal> normals;
    
    // Генерируем направления из вершин выпуклой оболочки
    for (int i = 0; i < h; i++) {
        for (int j = i + 1; j < h; j++) {
            long long dx = hull[j].x - hull[i].x;
            long long dy = hull[j].y - hull[i].y;
            
            long long g = gcd(dx, dy);
            if (g > 0) {
                dx /= g;
                dy /= g;
            }
            
            normals.insert({dx, dy});
            normals.insert({-dy, dx});
        }
    }
    
    double max_width = -1;
    long long best_a1, best_b1, best_c1;
    long long best_a2, best_b2, best_c2;
    
    for (auto [a, b] : normals) {
        if (a == 0 && b == 0) continue;
        
        long long min_val = LLONG_MAX;
        long long max_val = LLONG_MIN;
        
        for (const auto& p : points) {
            long long val = a * p.x + b * p.y;
            min_val = min(min_val, val);
            max_val = max(max_val, val);
        }
        
        double width = (max_val - min_val) / sqrt(a * a + b * b);
        
        if (width > max_width) {
            max_width = width;
            long long c1 = -min_val;
            long long c2 = -max_val;
            
            long long g1 = (c1 != 0) ? gcd3(a, b, c1) : gcd(a, b);
            if (g1 > 0) {
                best_a1 = a / g1;
                best_b1 = b / g1;
                best_c1 = c1 / g1;
            } else {
                best_a1 = a;
                best_b1 = b;
                best_c1 = c1;
            }
            
            long long g2 = (c2 != 0) ? gcd3(a, b, c2) : gcd(a, b);
            if (g2 > 0) {
                best_a2 = a / g2;
                best_b2 = b / g2;
                best_c2 = c2 / g2;
            } else {
                best_a2 = a;
                best_b2 = b;
                best_c2 = c2;
            }
        }
    }
    
    cout << best_a1 << " " << best_b1 << " " << best_c1 << "\n";
    cout << best_a2 << " " << best_b2 << " " << best_c2 << "\n";
    
    return 0;
}